//-----------------------------------------------------------------------------
//Copyright  2000-2003 TogetherSoft Corporation, a wholly owned subsidiary of Borland Software Corporation. All rights reserved. All patents pending.
//-----------------------------------------------------------------------------

using System;
using System.Collections;

namespace CSCashSales.data_objects
{
	/// <summary> 
	/// This is the class used to hold the description of one sale.
	/// It implements ICashSale and IPresentView interfaces.
	/// </summary>
	/// <TgData>
	/// </TgData>
    public class CashSale : ICashSale, IPresentView
    {
		// Codes of table attributes for IPresentView.

    /// <summary>
    /// Code ATTR_ITEM
    /// </summary>
		public const int ATTR_ITEM		= SalePosition.ATTR_ITEM;
    /// <summary>
    /// Code ATTR_NAME
    /// </summary>
		public const int ATTR_NAME		= SalePosition.ATTR_NAME;
    /// <summary>
    /// Code ATTR_UNIT
    /// </summary>
		public const int ATTR_UNIT		= SalePosition.ATTR_UNIT;
    /// <summary>
    /// Code ATTR_QUANTITY
    /// </summary>
		public const int ATTR_QUANTITY	= SalePosition.ATTR_QUANTITY;
    /// <summary>
    /// Code ATTR_PRICE
    /// </summary>
		public const int ATTR_PRICE		= SalePosition.ATTR_PRICE;

		private const int ATTR_COUNT = 5;

		// Codes of linear attributes for IPresentView.
    /// <summary>
    /// Code LATTR_SUBTOTAL
    /// </summary>
		public const int LATTR_SUBTOTAL = 0;
    /// <summary>
    /// Code LATTR_TAX
    /// </summary>
		public const int LATTR_TAX = 1;
    /// <summary>
    /// Code LATTR_TOTAL
    /// </summary>
		public const int LATTR_TOTAL = 2;
    /// <summary>
    /// Code LATTR_PAYMENT
    /// </summary>
		public const int LATTR_PAYMENT = 3;
    /// <summary>
    /// Code LATTR_CHANGE
    /// </summary>
		public const int LATTR_CHANGE = 4;

		private const int LATTR_COUNT = 5;

		/// <summary>
		/// Unique ID.
		/// </summary>
		private int m_intID;
		/// <summary>
		/// The sale date/time.
		/// </summary>
		private DateTime m_dtDate;
		/// <summary>
		/// The payment amount.
		/// </summary>
		private Decimal m_dcmPayment;
		/// <summary>
		/// A flag indicating whether total was completed.
		/// </summary>
		private bool m_blnHasTotal;
		/// <summary>
		/// A flag indicating whether payment was completed.
		/// </summary>
		private bool m_blnHasPayment;
		/// <summary>
		/// A container of sale positions.
		/// </summary>
		private ArrayList m_objList;
        
		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="intID"> The sale ID </param>
		/// <param name="dtDate"> The sale date </param>
    public CashSale( int intID, DateTime dtDate )
    {
			m_intID = intID;
			m_dtDate = dtDate;
			m_dcmPayment = new Decimal(0);
			m_blnHasTotal = false;
			m_blnHasPayment = false;
			m_objList = new ArrayList();
    }

		/// <summary>
		/// Returns the sale ID.
		/// </summary>
		public int getID() 
		{
			return m_intID;
		}

		/// <summary>
		/// Returns a count of sale positions.
		/// </summary>
		public int getPositionCount() 
		{
			return m_objList.Count;
		}
		
		/// <summary>
		/// An access to sale positions by its indexes.
		/// </summary>
		/// <param name="index"> The position index </param>
    public ISalePosition getPositionAt( int index ) 
		{
			return	m_objList[index] as ISalePosition;
		}

		/// <summary>
		/// Adds new position to the sale.
		/// </summary>
		/// <param name="newPosition"> The sale position </param>
		public void addPosition( ISalePosition newPosition ) 
		{
			m_objList.Add(newPosition);
		}

		/// <summary>
		/// Returns the sale date.
		/// </summary>
		public DateTime getSaleDate() 
		{
			return m_dtDate;
		}

		/// <summary>
		/// Calculates the sale subtotal.
		/// </summary>
		public Decimal getSubTotal() 
		{
			Decimal dcmSum = new Decimal(0);
			for(int i=0; i<getPositionCount(); i++)
			{
				ISalePosition position = getPositionAt(i);
				dcmSum += position.getAmount();
			}
			return dcmSum;
		}

		/// <summary>
		/// Calculates the sale tax.
		/// </summary>
		public Decimal getTax() 
		{
			return getSubTotal() * new Decimal(0.06);
		}

		/// <summary>
		/// Calculates the sale total.
		/// </summary>
		public Decimal getTotal() 
		{
			return getSubTotal() + getTax();
		}

		/// <summary>
		/// Returns the payment.
		/// </summary>
		public Decimal getPayment() 
		{
			return m_dcmPayment;
		}

		/// <summary>
		/// Performs the payment operation.
		/// </summary>
		/// <param name="payment"> The payment amount </param>
		public bool setPayment( Decimal payment ) 
		{
			bool blnResult = false;
			if(isPaymentValid(payment))
			{
				m_dcmPayment = payment;
				m_blnHasPayment = true;
				blnResult = true;
			}
			return blnResult;
		}

		/// <summary>
		/// Returns the sale change.
		/// </summary>
		public Decimal getChange() 
		{
			return getPayment() - getTotal();
		}

		/// <summary>
		/// Indicates whether the total operation is available.
		/// </summary>
		public bool isTotalAvailable() 
		{
			return (getPositionCount() > 0) && (!hasTotal());
		}

		/// <summary>
		/// Indicates whether the payment operation is available.
		/// </summary>
		public bool isPaymentAvailable() 
		{
			return hasTotal() && (!hasPayment());
		}

		/// <summary>
		/// Performs the total operation.
		/// </summary>
		public bool makeTotal() 
		{
			if(!hasTotal())
			{
				m_blnHasTotal = true;
				return true;
			}
			else
				return false;
		}

		/// <summary>
		/// Indicates whether the total was completed.
		/// </summary>
		public bool hasTotal() 
		{
			return m_blnHasTotal;
		}

		/// <summary>
		/// Indicates whether the payment was completed.
		/// </summary>
		public bool hasPayment() 
		{
			return m_blnHasPayment;
		}
	
		/// <summary>
		/// Indicates whether the sale is completed and ready to save.
		/// </summary>
		public bool isValid() 
		{
			return hasTotal() && hasPayment();
		}

		/// <summary>
		/// Checks whether the specified amount is valid to perform the payment.
		/// </summary>
		/// <param name="payment"> The amount </param>
		public bool isPaymentValid( Decimal payment ) 
		{
			return (payment > 0) && (getTotal() <= payment);
		}

		/// <summary>
		/// Indicates whether the object has linear presentable attributes.
		/// </summary>
		public bool hasLinearAttributes() 
		{
			return true;
		}

		/// <summary>
		/// Indicates whether the object has table presentable attributes.
		/// </summary>
		public bool hasTableAttributes() 
		{
			return true;
		}

		/// <summary>
		/// Returns the number of linear presentable attributes.
		/// </summary>
		public int getLinearAttrCount() 
		{
			return LATTR_COUNT;
		}

		/// <summary>
		/// Returns the name of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrName( int attr ) 
		{
			switch(attr)
			{
				case LATTR_SUBTOTAL:
					return "Subtotal";
				case LATTR_TAX:
					return "Tax";
				case LATTR_TOTAL:
					return "Total";
				case LATTR_PAYMENT:
					return "Payment";
				case LATTR_CHANGE:
					return "Change";
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the value of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrValue( int attr ) 
		{
			switch(attr)
			{
				case LATTR_SUBTOTAL:
					return Utils.AsCurrency(getSubTotal());
				case LATTR_TAX:
					return Utils.AsCurrency(getTax());
				case LATTR_TOTAL:
					return Utils.AsCurrency(getTotal());
				case LATTR_PAYMENT:
					return Utils.AsCurrency(getPayment());
				case LATTR_CHANGE:
					return Utils.AsCurrency(getChange());
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the number of table presentable attributes.
		/// </summary>
		public int getTableAttrCount() 
		{
			return ATTR_COUNT;
		}

		/// <summary>
		/// Returns the size of attribute table.
		/// </summary>
		public int getTableRowCount() 
		{
			return getPositionCount();	
		}

		/// <summary>
		/// Returns the name of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getTableAttrName( int attr ) 
		{
			switch(attr)
			{
				case ATTR_ITEM:
					return "Item";
				case ATTR_NAME:
					return "Name";
				case ATTR_UNIT:
					return "Unit";
				case ATTR_QUANTITY:
					return "Quantity";
				case ATTR_PRICE:
					return "Price";
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the value of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		/// <param name="row"> The table row </param>
		public String getTableAttrValue( int attr, int row ) 
		{
			SalePosition position = getPositionAt(row) as SalePosition;
			return position.getLinearAttrValue(attr);			
		}

    }
}
